<?php

namespace App\Http\Controllers\Customer;

use App\Http\Controllers\Controller;
use App\Imports\BlockContactsImport;
use App\Models\Contact;
use App\Models\CustomerPlan;
use App\Models\Group;
use App\Models\Label;
use App\Models\Plan;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Maatwebsite\Excel\Concerns\ToCollection;
use Maatwebsite\Excel\Concerns\WithCustomCsvSettings;
use Maatwebsite\Excel\Facades\Excel;


class BlockContactController extends Controller
{
    public function index(){
        return view('customer.contacts.block');
    }
    public function blockNumber($id,Request $request){

        $block_contact = Contact::find($id);
        $block_contact->block_contact = $request->type;
        $block_contact->save();

        return back()->with('success', 'Contact successfully Block');
    }
    public function getAll(Request $request)
    {
        $customer = auth('customer')->user();
        if($customer->type=='staff'){
            $customer=$customer->staff;
        }

        $contacts = $customer->contacts()->with('contact_groups','contact_groups.group')->where('block_contact','block');

        return datatables()->of($contacts)
        ->addColumn('full_name', function ($q) {
            return $q->full_name;
        })
            ->addColumn('group', function ($q) {
                $group_name='NA';
                $contact_group=$q->contact_groups;
                if(isset($contact_group[0]) && $contact_group[0]->group){
                    $group_name=$contact_group[0]->group->name;
                }
            return $group_name;
        })
         ->addColumn('action', function ($q) {
            return '<button class="btn btn-sm btn-info mr-2" data-message="Are you sure you want to  Unblock this number?"
                                        data-action=' . route('customer.block.number', [$q,'type'=>'unblock']) . '
                                        data-input={"_method":"get"}
                                        data-toggle="modal" data-target="#modal-confirm" data-toggle="tooltip" data-placement="top" title="Unblock"><i class="fa fa-undo"></i></button>';
            })
         ->filterColumn('full_name', function($query, $keyword) {
                $sql = "CONCAT(first_name,' ',last_name)  like ?";
                    $query->whereRaw($sql, ["%{$keyword}%"])->orWhereRaw('first_name like ?', ["%{$keyword}%"])->orWhereRaw('last_name like ?', ["%{$keyword}%"]);
            })
            ->filterColumn('group', function ($query, $keyword) {
                $query->whereHas('contact_groups.group', function ($q) use ($keyword) {
                    $q->where('name', 'like', "%" . $keyword . "%");
                });
            })

            ->rawColumns(['action','group'])
            ->toJson();
    }
    public function import_block_contacts()
    {
        return view('customer.contacts.block_import');
    }
    public function import_block_contacts_store(Request $request)
    {
        $request->validate([
            'import_name' => 'required',
            'import_contact_csv' => 'required|mimes:csv,txt'
        ]);
        $customer = auth('customer')->user();
        if($customer->type=='staff'){
            $customer=$customer->staff;
        }
        $currentPlan = $customer->currentPlan();
        if (isset($currentPlan->renew_date) && $currentPlan->renew_date < Carbon::now()){
            return back()->with('fail', 'Your Plan has expired');
        }
        DB::beginTransaction();
        try {
            $preGroup = $customer->groups()->where('name', $request->import_name)->first();
            if ($preGroup) return back()->withErrors(['msg' => "Import name already exists"]);

            // $importContact = new Group();
            // $importContact->customer_id = auth('customer')->id();
            // $importContact->name = $request->import_name;
            // $importContact->save();

            $group_name= $request->import_name ;


            if ($request->hasFile('import_contact_csv')) {
                $data = $request->file('import_contact_csv');
                $fileName = $customer->id . '.' . $data->getClientOriginalExtension();
                $data->move(public_path() . '/uploads', $fileName);
                $file_url = public_path() . '/uploads/' . $fileName;
                $fp = file($file_url);
                $contacts = $customer->contacts()->count();
                $importedContact = (count($fp) - 1) + $contacts;
                $current_plan = $customer->currentPlan();

                if (!$current_plan)
                    throw new \ErrorException('Doesn\'t have any plan right now');
                $planContactLimit = $current_plan->contact_limit;
                if ($importedContact >= $planContactLimit) {
                    return redirect()->route('customer.block.contacts')->with('fail', 'You have extended your contact limit');
                }
                try {
                    Excel::import(new BlockContactsImport($group_name, $customer), $file_url);
                    DB::commit();
                } catch (\Exception $ex) {
                    if (isset($ex->validator)) {
                        return redirect()->back()->withErrors($ex->validator->errors());
                    } else {
                        return redirect()->back()->withErrors(['msg' => $ex->getMessage()]);
                    }

                }
            }

        } catch (\Exception $ex) {
            Log::error($ex);
            DB::rollBack();
            return back()->with('fail', $ex->getMessage());
        }

        return back()->with('success', 'Import Block Contact Successfully Created');
    }
}
